# 画面設計書 6-Real-Time Decryption

## 概要

本ドキュメントは、VBCorLibプロジェクトのサンプルアプリケーション「Real-Time Decryption」の画面設計書です。

### 本画面の処理概要

Real-Time Decryptionは、VBCorLibライブラリが提供するCryptoStreamクラスを使用して、暗号化ファイルのリアルタイム復号機能をデモンストレーションする画面です。従来の方法ではファイル全体をメモリに読み込んで復号する必要がありましたが、CryptoStreamを使用することで、ファイルをストリームとして読み込みながら必要な部分だけをその場で（オンザフライで）復号できることを示しています。

**業務上の目的・背景**：暗号化されたデータを扱う際、メモリ効率と処理効率が重要な課題となります。CryptoStreamは暗号化/復号処理をストリーム操作として抽象化し、大容量ファイルでも効率的に処理できます。本画面は、プレーンテキストファイルと暗号化ファイルを同じコードで読み込む方法を示し、開発者に暗号化ストリームの利点を伝えることを目的としています。

**画面へのアクセス方法**：Examples/Cryptography/RealTimeDecryption/RealTimeDecryption.vbpプロジェクトを開き、F5キーで実行することでこの画面が表示されます。

**主要な操作・処理内容**：
1. フォーム起動時にプレーンテキストファイルと暗号化ファイルを自動生成
2. 「Read Line」ボタンで両ファイルから1行ずつ読み込み
3. txtPlainTextにプレーンテキストの内容を追加表示
4. txtCipherTextに復号された内容を追加表示
5. ファイル終端で「Finished」を表示

**画面遷移**：本画面は独立したデモアプリケーションであり、他の画面への遷移はありません。

**権限による表示制御**：本画面は開発者向けデモツールであり、特別な権限制御は実装されていません。

## 関連機能

| 機能No | 機能名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| 70 | CryptoStream | 主機能 | リアルタイム暗号化/復号ストリーム（暗号化ファイルのオンザフライ復号） |
| 49 | RijndaelManaged | 主機能 | Rijndael暗号化/復号化（CreateEncryptor、CreateDecryptor） |
| 28 | StreamReader | 補助機能 | 暗号化ストリームからのテキスト読み込み（ReadLine） |
| 29 | StreamWriter | 補助機能 | 暗号化ストリームへのテキスト書き込み（WriteLine） |
| 26 | FileStream | 補助機能 | ファイルストリームの基盤（InputStream、OutputStream） |
| 34 | File | 補助機能 | ファイルオープン操作（OpenFile） |
| 39 | Path | 補助機能 | ファイルパスの結合（Combine） |
| 2 | CorString | 補助機能 | 文字列操作（IsNull、Format） |

## 画面種別

デモンストレーション / 暗号化処理

## URL/ルーティング

VB6デスクトップアプリケーションのため、URLルーティングは存在しません。フォームファイル: `Examples/Cryptography/RealTimeDecryption/Form1.frm`

## 入出力項目

本画面にユーザー入力項目はありません。ボタン操作のみです。

| 項目名 | コントロール | 項目種別 | データ型 | 説明 |
|--------|-------------|----------|----------|------|
| txtPlainText | TextBox | 出力 | String | プレーンテキストファイルの内容表示（MultiLine） |
| txtCipherText | TextBox | 出力 | String | 復号されたテキストの内容表示（MultiLine） |

## 表示項目

| 項目名 | コントロール | データ型 | 説明 |
|--------|-------------|----------|------|
| txtPlainText | TextBox | String | PlainTextLines.txtから読み込んだ行を追加表示 |
| txtCipherText | TextBox | String | EncryptedTextLines.txtを復号した行を追加表示 |

## イベント仕様

### 1-Form_Load

フォーム読み込み時にソースファイル作成とストリーム初期化を行います。

**処理内容**:
1. CreateSourceFiles - プレーンテキストファイルと暗号化ファイルを作成
2. OpenSourceFiles - 両ファイルのストリームを開く

**コード参照**: 行88-91

```vb
Private Sub Form_Load()
    Call CreateSourceFiles
    Call OpenSourceFiles
End Sub
```

### 2-cmdReadLine_Click

両ファイルから1行ずつ読み込んで表示します。

**処理内容**:
1. プレーンテキストが完了していなければReadLineを呼び出し
2. 暗号化テキストが完了していなければReadLineを呼び出し

**コード参照**: 行83-86

```vb
Private Sub cmdReadLine_Click()
    If Not mPlainTextFinished Then mPlainTextFinished = ReadLine(mTextReader, txtPlainText)
    If Not mEncryptedFinished Then mEncryptedFinished = ReadLine(mEncryptedReader, txtCipherText)
End Sub
```

### 3-ReadLine（内部関数）

ストリームから1行読み込んでテキストボックスに追加します。

**処理内容**:
1. Reader.ReadLineで1行読み込み
2. 行が有効ならテキストボックスに追加
3. ファイル終端なら「Finished」を追加してTrueを返す

**コード参照**: 行95-106

### 4-OpenSourceFiles（内部関数）

ソースファイルを開いてストリームを初期化します。

**処理内容**:
1. プレーンテキストファイルをStreamReaderで開く
2. 暗号化ファイルをFileStream→CryptoStream→StreamReaderでラップして開く

**コード参照**: 行110-129

```vb
Private Sub OpenSourceFiles()
    ' Opening the plain text file is simple.
    Set mTextReader = NewStreamReader(Path.Combine(App.Path, "PlainTextLines.txt"))

    ' Opening the encrypted file takes some extra steps.
    Dim InputStream As FileStream
    Set InputStream = File.OpenFile(Path.Combine(App.Path, "EncryptedTextLines.txt"), FileMode.OpenExisting)

    ' Wrap the FileStream with our decrypting CryptoStream.
    Dim DecryptStream As CryptoStream
    Set DecryptStream = NewCryptoStream(InputStream, mCipher.CreateDecryptor, CryptoStreamMode.ReadMode)

    ' Place CryptoStream in a StreamReader
    Set mEncryptedReader = NewStreamReader(DecryptStream)
End Sub
```

### 5-CreateSourceFiles（内部関数）

プレーンテキストファイルと暗号化ファイルを作成します。

**処理内容**:
1. CreatePlainTextFile - 10行のプレーンテキストファイル作成
2. CreateEncryptedFile - 10行の暗号化ファイル作成

**コード参照**: 行131-134

### 6-CreateEncryptedFile（内部関数）

暗号化ファイルを作成します。CryptoStreamを使用してオンザフライで暗号化します。

**処理内容**:
1. FileStreamで出力ファイルを開く
2. CryptoStreamでEncryptorをラップ
3. WriteLines関数で10行書き込み

**コード参照**: 行163-188

## データベース更新仕様

本画面はデータベースを使用しません。

## メッセージ仕様

| メッセージID | メッセージ内容 | 表示条件 |
|-------------|---------------|----------|
| MSG-01 | "Finished" | ファイル読み込み完了時 |

## 例外処理

| 例外種別 | 対処方法 |
|---------|---------|
| ファイル作成エラー | VBランタイムエラー発生 |
| 復号エラー | CryptoStreamがエラーを発生 |

## 備考

- BorderStyle = Fixed Single（サイズ変更不可）
- MaxButton = False, MinButton = False（最大化・最小化ボタン無効）
- RijndaelManagedクラスをNew演算子で初期化（デフォルト鍵を使用）
- 生成ファイル: PlainTextLines.txt, EncryptedTextLines.txt
- CryptoStreamMode.ReadModeで復号ストリーム、WriteModeで暗号化ストリーム

---

## コードリーディングガイド

本画面を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

暗号化処理で使用されるオブジェクトの関係を理解します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | Form1.frm | `Examples/Cryptography/RealTimeDecryption/Form1.frm` | 行69-81: モジュールレベル変数宣言 |

**読解のコツ**: StreamReaderが2つ宣言されていますが、一方は直接FileStreamを読み、もう一方はCryptoStreamをラップしたFileStreamを読みます。

```vb
Private mTextReader As StreamReader       ' プレーンテキスト用
Private mEncryptedReader As StreamReader  ' 暗号化ファイル用（CryptoStream経由）
Private mCipher As New RijndaelManaged    ' 暗号化エンジン
```

#### Step 2: エントリーポイントを理解する

フォーム読み込み時の初期化処理がエントリーポイントです。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | Form1.frm | `Examples/Cryptography/RealTimeDecryption/Form1.frm` | 行88-91: Form_Load |

**主要処理フロー**:
1. **行89**: CreateSourceFiles - ファイル作成
2. **行90**: OpenSourceFiles - ストリーム初期化

#### Step 3: 暗号化ファイル作成処理を理解する

CryptoStreamを使用した暗号化書き込みを確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | Form1.frm | `Examples/Cryptography/RealTimeDecryption/Form1.frm` | 行163-188: CreateEncryptedFile |

**主要処理フロー（暗号化書き込み）**:
- **行171-172**: FileStreamで出力ファイルを開く
- **行182**: NewCryptoStream(OutputStream, mCipher.CreateEncryptor, CryptoStreamMode.WriteMode)
- **行187**: WriteLines関数でCryptoStreamに書き込み

#### Step 4: 復号読み込み処理を理解する

CryptoStreamを使用した復号読み込みを確認します。

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | Form1.frm | `Examples/Cryptography/RealTimeDecryption/Form1.frm` | 行110-129: OpenSourceFiles |

**主要処理フロー（復号読み込み）**:
- **行117**: File.OpenFileでFileStream取得
- **行124**: NewCryptoStream(InputStream, mCipher.CreateDecryptor, CryptoStreamMode.ReadMode)
- **行128**: NewStreamReader(DecryptStream)でテキスト読み込み可能に

### プログラム呼び出し階層図

```
Form_Load
    │
    ├─ CreateSourceFiles
    │      │
    │      ├─ CreatePlainTextFile
    │      │      └─ File.OpenFile → StreamWriter → WriteLines
    │      │
    │      └─ CreateEncryptedFile
    │             └─ File.OpenFile → CryptoStream(Encryptor) → WriteLines
    │
    └─ OpenSourceFiles
           │
           ├─ NewStreamReader(PlainTextLines.txt)
           │      └─ mTextReader
           │
           └─ File.OpenFile(EncryptedTextLines.txt)
                  │
                  └─ NewCryptoStream(Decryptor)
                         │
                         └─ NewStreamReader
                                └─ mEncryptedReader

cmdReadLine_Click
    │
    ├─ ReadLine(mTextReader, txtPlainText)
    │      └─ Reader.ReadLine → テキスト追加
    │
    └─ ReadLine(mEncryptedReader, txtCipherText)
           └─ Reader.ReadLine → [CryptoStream復号] → テキスト追加
```

### データフロー図

```
[暗号化書き込み]
テキスト行 ───▶ StreamWriter ───▶ CryptoStream ───▶ FileStream ───▶ ファイル
                                   (Encryptor)

[復号読み込み]
ファイル ───▶ FileStream ───▶ CryptoStream ───▶ StreamReader ───▶ テキスト行
                              (Decryptor)

[比較表示]
PlainTextLines.txt ─────────▶ StreamReader ─────────────▶ txtPlainText

EncryptedTextLines.txt ─────▶ CryptoStream(Decrypt) ────▶ txtCipherText
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| Form1.frm | `Examples/Cryptography/RealTimeDecryption/Form1.frm` | ソース | メインフォーム |
| RealTimeDecryption.vbp | `Examples/Cryptography/RealTimeDecryption/RealTimeDecryption.vbp` | プロジェクト | プロジェクト定義 |
| PlainTextLines.txt | 実行時生成 | データ | プレーンテキストファイル |
| EncryptedTextLines.txt | 実行時生成 | データ | 暗号化ファイル |
| VBCorLib.dll | 外部 | DLL | VBCorLibライブラリ本体 |
